<?php

namespace App\supplier1;

use App\Services\LogManager;

class EcoCategoryComparison {
    private $logManager;
    private $inputDir;
    private $outputDir;
    private $categoryExtrasFile;
    private $foldersInitialized = false;

    public function __construct() {
        $this->logManager = new LogManager();
        $this->inputDir = __DIR__ . '/../../data/supplier1/';
        $this->categoryExtrasFile = $this->inputDir . 'Category-Website-Extras.csv';
    }

    // Call this method only when folder setup is needed
    private function setupDirectories() {
        if ($this->foldersInitialized) {
            return;  // Prevent repeated initialization
        }

        // Set up directories for output specific to Supplier 1
        $this->outputDir = __DIR__ . '/../../data-extracted/supplier1/eco-category-comparison/eco-category-comparison-' . date("Y-m-d-H-i-s") . '/';

        // Ensure the output directory exists, if not create it
        if (!is_dir($this->outputDir)) {
            mkdir($this->outputDir, 0777, true);
            $this->logManager->logInfo("Created output directory: " . $this->outputDir, 'ecocomparison', 'supplier1');
        }

        $this->foldersInitialized = true;  // Mark as initialized
    }

    public function compareEcoCategories() {
        // Ensure the directories are set up only when necessary
        $this->setupDirectories();

        if (!file_exists($this->categoryExtrasFile)) {
            $this->logManager->logError("Category extras file does not exist: " . $this->categoryExtrasFile, 'ecocomparison', 'supplier1');
            throw new \Exception("Category extras file does not exist: " . $this->categoryExtrasFile);
        }

        $catExtrasArr = $this->createCatArray($this->categoryExtrasFile);

        // Initialize arrays if not present
        if (!isset($catExtrasArr['Environmental'])) {
            $catExtrasArr['Environmental'] = [];
        }

        // Path to the eco categories CSV file
        $ecoFile = $this->getLatestEcoCategoryFile();

        if (file_exists($ecoFile)) {
            $ecoCategories = $this->csvColumnToArrayNoHeaders($ecoFile, ',', 0);

            $this->logManager->logInfo("Comparing eco categories...", 'ecocomparison', 'supplier1');

            $additions = [];
            $removals = [];

            foreach ($ecoCategories as $ecoCategory) {
                $ecoCategory = $this->hyperify($ecoCategory);
                if (!in_array($ecoCategory, $catExtrasArr['ECO'])) {
                    $additions[] = $ecoCategory;
                }
            }

            foreach ($catExtrasArr['ECO'] as $category) {
                if (!in_array($this->hyperify($category), $ecoCategories)) {
                    $removals[] = $category;
                }
            }

            $this->saveEcoCategoryAdditions($additions);
            $this->saveEcoCategoryRemovals($removals);

            // After generating the CSVs, update Category-Website-Extras.csv
            $this->updateCategoryExtrasFile($catExtrasArr, $additions, $removals);

            $this->logManager->logInfo("Eco category comparison completed.", 'ecocomparison', 'supplier1');
        } else {
            $this->logManager->logError("Eco categories file does not exist: " . $ecoFile, 'ecocomparison', 'supplier1');
            throw new \Exception("Eco categories file does not exist: " . $ecoFile);
        }
    }

    private function getLatestEcoCategoryFile() {
        $splitterDir = __DIR__ . '/../../data-extracted/supplier1/category-splitter/';
        $latestDir = null;
        $latestTime = 0;

        if (is_dir($splitterDir)) {
            $dirs = scandir($splitterDir);
            foreach ($dirs as $dir) {
                if ($dir === '.' || $dir === '..') continue;

                $dirPath = $splitterDir . $dir;
                if (is_dir($dirPath)) {
                    $dirTime = filemtime($dirPath);
                    if ($dirTime > $latestTime) {
                        $latestTime = $dirTime;
                        $latestDir = $dirPath;
                    }
                }
            }
        }

        if ($latestDir) {
            $latestFile = $latestDir . '/eco-categories.csv';
            if (file_exists($latestFile)) {
                return $latestFile;
            } else {
                throw new \Exception("Eco categories file does not exist in the latest directory: " . $latestFile);
            }
        } else {
            throw new \Exception("No directories found in category-splitter");
        }
    }

    private function saveEcoCategoryAdditions($additions) {
        $filePath = $this->outputDir . 'eco-category-additions.csv';
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'ecocomparison', 'supplier1');
            return;
        }
        foreach ($additions as $addition) {
            fputcsv($file, [$addition]);
        }
        fclose($file);
        $this->logManager->logInfo("Saved eco category additions to CSV: $filePath", 'ecocomparison', 'supplier1');
    }

    private function saveEcoCategoryRemovals($removals) {
        $filePath = $this->outputDir . 'eco-category-removals.csv';
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'ecocomparison', 'supplier1');
            return;
        }
        foreach ($removals as $removal) {
            fputcsv($file, [$removal]);
        }
        fclose($file);
        $this->logManager->logInfo("Saved eco category removals to CSV: $filePath", 'ecocomparison', 'supplier1');
    }

    // Update Category-Website-Extras.csv with additions and removals
    private function updateCategoryExtrasFile(&$catExtrasArr, $additions, $removals) {
        // Remove categories
        $catExtrasArr['ECO'] = array_diff($catExtrasArr['ECO'], $removals);
        // Add new categories
        $catExtrasArr['ECO'] = array_merge($catExtrasArr['ECO'], $additions);

        // Rebuild the CSV structure
        $this->saveCategoryExtras($catExtrasArr);
    }

    // Save the updated Category-Website-Extras.csv
    private function saveCategoryExtras($categoryData) {
        $filePath = $this->categoryExtrasFile;
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'ecocomparison', 'supplier1');
            return;
        }

        // Write main categories as the first row
        fputcsv($file, array_keys($categoryData));

        // Write subcategories under each main category
        $maxRows = max(array_map('count', $categoryData));
        for ($i = 0; $i < $maxRows; $i++) {
            $row = [];
            foreach ($categoryData as $subCategories) {
                $row[] = isset($subCategories[$i]) ? $subCategories[$i] : '';
            }
            fputcsv($file, $row);
        }

        fclose($file);
        $this->logManager->logInfo("Updated Category-Website-Extras.csv successfully.", 'ecocomparison', 'supplier1');
    }

    // Non optimized one
    // private function createCatArray($csvFile = '', $delimiter = ',') {
    //     if (!file_exists($csvFile) || !is_readable($csvFile)) {
    //         exit("Your file doesn't exist or is not readable");
    //     }

    //     $header = null;
    //     $dataset = [];
    //     $delimiter = ",";

    //     if (($handle = fopen($csvFile, 'r')) !== false) {
    //         while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
    //             if (!$header) {
    //                 $header = $row;
    //             } else {
    //                 $x = 0;
    //                 foreach ($row as $cell) {
    //                     if ($cell) {
    //                         $cell = trim($cell);
    //                         $dataset[$header[$x]][] = $cell;
    //                     }
    //                     $x++;
    //                 }
    //             }
    //         }
    //         fclose($handle);
    //     }
    //     return $dataset;
    // }

    private function createCatArray($csvFile = '', $delimiter = ',') {
        if (!file_exists($csvFile) || !is_readable($csvFile)) {
            exit("Your file doesn't exist or is not readable");
        }
    
        $file = new \SplFileObject($csvFile);
        $file->setFlags(\SplFileObject::READ_CSV);
        $file->setCsvControl($delimiter);
    
        $header = null;
        $dataset = [];
        foreach ($file as $row) {
            if ($file->key() === 0) {
                $header = $row;
            } else {
                foreach ($row as $x => $cell) {
                    if ($cell) {
                        $dataset[$header[$x]][] = trim($cell);
                    }
                }
            }
        }
        return $dataset;
    }

    // NonOptimized one
    // private function csvColumnToArrayNoHeaders($file = '', $delimiter = ',', $column = '0') {
    //     if (!file_exists($file) || !is_readable($file)) {
    //         throw new \Exception($file . " doesn't exist or is not readable.");
    //     }

    //     $colArray = [];
    //     if (($handle = fopen($file, 'r')) !== false) {
    //         while (($data = fgetcsv($handle, 1000, $delimiter)) !== false) {
    //             $colArray[] = $data[$column];
    //         }
    //         fclose($handle);
    //     }
    //     return $colArray;
    // }

    private function csvColumnToArrayNoHeaders($file = '', $delimiter = ',', $column = '0') {
        if (!file_exists($file) || !is_readable($file)) {
            throw new \Exception($file . " doesn't exist or is not readable.");
        }
    
        $file = new \SplFileObject($file);
        $file->setFlags(\SplFileObject::READ_CSV);
        $file->setCsvControl($delimiter);
    
        $colArray = [];
        foreach ($file as $row) {
            $colArray[] = $row[$column] ?? null;
        }
        return array_filter($colArray);
    }

    private function hyperify($data) {
        $data = strtolower($data);

        if (preg_match("/-/", $data)) {
            // Hyphens
            $result = explode("-", $data);
            foreach ($result as $value) {
                $return[] = ucfirst(strtolower($value));
            }
            $return = implode("-", $return);
        } else {
            // All the other exceptions
            $data = ucwords($data);
            $data = str_replace("And", "and", $data);
            $data = str_replace("&", "and", $data);
            $data = str_replace("Ipad", "iPad", $data);
            $data = str_replace("Iphone", "iPhone", $data);
            $data = str_replace("Rfid", "RFID", $data);
            $data = str_replace("Usb", "USB", $return = $data);
        }

        return $return;
    }
}
