<?php

namespace App\supplier1;

use App\Services\LogManager;

class DataChecker {
    private $logManager;
    private $dataDir;
    private $foldersInitialized = false; // Flag to check if directories are set up

    public function __construct() {
        $this->logManager = new LogManager();
    }

    // Move the directory setup logic here
    private function setupDataDirectory() {
        if ($this->foldersInitialized) {
            return;  // Prevent repeated initialization
        }

        // Set the base directory specifically for Supplier 1
        $this->dataDir = __DIR__ . '/../../data-extracted/supplier1/invalid-categories';
        
        if (!is_dir($this->dataDir)) {
            mkdir($this->dataDir, 0777, true);  // Create the directory if it does not exist
            $this->logManager->logInfo("Created data directory: " . $this->dataDir, 'data', 'supplier1');
        }

        $this->foldersInitialized = true; // Mark directories as initialized
    }

    public function checkData($products) {
        // Ensure the data directory is set up before saving the file
        $this->setupDataDirectory();
        
        $invalidProducts = [];
        foreach ($products as $product) {
            if ($this->hasDataIssues($product)) {
                $invalidProducts[] = $product;
            }
        }
        if (!empty($invalidProducts)) {
            $this->saveInvalidProductsToCsv($invalidProducts);
            return true; // Return true if there are invalid products
        }
        return false; // Return false if no invalid products are found
    }
    

    private function hasDataIssues($product) {
        $categorySplit = explode("/", $product['category']);
        $split0 = isset($categorySplit[0]) ? trim($categorySplit[0]) : '';
        $split1 = isset($categorySplit[1]) ? trim($categorySplit[1]) : '';

        return $split1 !== trim($product['index_entry']);
    }

    private function saveInvalidProductsToCsv($products) {
        $timestamp = date("Y-m-d-H-i-s");
        $filePath = "{$this->dataDir}/invalid-categories-{$timestamp}.csv";

        $file = fopen($filePath, 'w');  // Open a new file for writing
        if (!$file) {
            $this->logManager->logError("Failed to open file: $filePath", 'data', 'supplier1');
            return;
        }

        // Write the CSV headers
        fputcsv($file, ['Code', 'Product Name', 'Category', 'Split0', 'Split1', 'Index']);
        foreach ($products as $product) {
            $categorySplit = explode("/", $product['category']);
            $split0 = isset($categorySplit[0]) ? trim($categorySplit[0]) : '';
            $split1 = isset($categorySplit[1]) ? trim($categorySplit[1]) : '';

            // Write each product's data to the CSV
            fputcsv($file, [
                $product['code'],
                $product['title'],
                $product['category'],
                $split0,
                $split1,
                $product['index_entry']
            ]);
        }
        fclose($file);  // Close the file after writing
        $this->logManager->logInfo("Invalid products saved to CSV: $filePath", 'data', 'supplier1');
    }
}
