<?php

namespace App\supplier1;

use App\Services\LogManager;

class CategorySplitter {
    private $logManager;
    private $inputDir;
    private $outputDir;
    private $foldersInitialized = false; // Flag to ensure directories are only set up once

    public function __construct() {
        $this->logManager = new LogManager();
        $this->inputDir = __DIR__ . '/../../data/supplier1/';
    }

    // Set up directories only when necessary
    private function setupDirectories() {
        if ($this->foldersInitialized) {
            return; // Prevent repeated initialization
        }

        // Set up the output directory for category splitter
        $this->outputDir = __DIR__ . '/../../data-extracted/supplier1/category-splitter/category-splitter-' . date("Y-m-d-H-i-s") . '/';

        // Ensure the output directory exists, if not create it
        if (!is_dir($this->outputDir)) {
            mkdir($this->outputDir, 0777, true);
            $this->logManager->logInfo("Created output directory: " . $this->outputDir, 'categorysplitter', 'supplier1');
        }

        // Check if the input directory exists
        if (!is_dir($this->inputDir)) {
            $this->logManager->logError("Input directory does not exist: " . $this->inputDir, 'categorysplitter', 'supplier1');
            throw new \Exception("Input directory does not exist: " . $this->inputDir);
        }

        $this->foldersInitialized = true; // Mark as initialized
    }

    public function splitCategories($products) {
        // Ensure directories are set up before splitting categories
        $this->setupDirectories();

        $outputFiles = [
            'primary-categories.csv',
            'secondary-categories.csv',
            'eco-categories.csv'
        ];

        // Remove old files
        foreach ($outputFiles as $file) {
            $filePath = $this->outputDir . $file;
            if (file_exists($filePath)) {
                unlink($filePath);
                $this->logManager->logInfo("Removed $filePath", 'categorysplitter', 'supplier1');
            }
        }

        $catPrimary = [];
        $catSecondary = [];
        $ecoSubCat = [];

        foreach ($products as $product) {
            $title = str_replace('"', "”", $product['title']); // Just replace straight quotes
            $categorySplit = explode("/", $product['category']);

            $title = rtrim($title);
            $product['category'] = rtrim($product['category']);
            $split0 = isset($categorySplit[0]) ? rtrim($categorySplit[0]) : '';
            $split1 = isset($categorySplit[1]) ? rtrim($categorySplit[1]) : '';

            $catPrimary[] = $split0;
            $catSecondary[] = $split1;

            if ($split0 == "Environmental") {
                $ecoSubCat[] = $split1;
            }
        }

        // Save the unique values to CSV files
        $this->saveUniqueValuesToCsv(array_unique($catPrimary), $this->outputDir . 'primary-categories.csv');
        $this->saveIndexedValuesToCsv(array_unique($catSecondary), $this->outputDir . 'secondary-categories.csv');
        $this->saveUniqueValuesToCsv(array_unique($ecoSubCat), $this->outputDir . 'eco-categories.csv');
    }

    private function saveUniqueValuesToCsv($values, $filePath) {
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'categorysplitter', 'supplier1');
            return;
        }
        foreach ($values as $value) {
            fputcsv($file, [$value]);
        }
        fclose($file);
        $this->logManager->logInfo("Saved unique values to CSV: $filePath", 'categorysplitter', 'supplier1');
    }

    private function saveIndexedValuesToCsv($values, $filePath) {
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'categorysplitter', 'supplier1');
            return;
        }
        $index = 100;
        foreach ($values as $value) {
            fputcsv($file, [(string)$index, $value]);
            $index++;
        }
        fclose($file);
        $this->logManager->logInfo("Saved indexed values to CSV: $filePath", 'categorysplitter', 'supplier1');
    }
}
