<?php

namespace App\Api;

use GuzzleHttp\Client;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Promise;
use GuzzleHttp\Handler\CurlMultiHandler;
use GuzzleHttp\Exception\RequestException;
use Exception;

class OpenCartClient {
    private $client;
    private $accessToken;

    public function __construct() {
        // Use CurlMultiHandler to enable persistent connections
        $handler = new CurlMultiHandler();
        $stack = HandlerStack::create($handler);
        
        // Initialize Guzzle client
        $this->client = new Client([
            'base_uri' => $_ENV['OPENCART_API_URL'],
            'handler' => $stack, // Set the handler to enable Keep-Alive
            'headers' => [
                'Connection' => 'keep-alive', // Enable Keep-Alive for persistent connections
            ],
            'timeout'  => 50.0,
        ]);

        // Authenticate and login during instantiation
        $this->authenticate();
        $this->login();
    }

    /**
     * Authenticate and fetch access token
     * @return void
     */
    private function authenticate() {
        try {
            $response = $this->client->post('oauth2/token/client_credentials', [
                'headers' => [
                    'Authorization' => 'Basic ' . $_ENV['AUTHORIZATION_KEY'], // The basic auth token from the credentials
                ]
            ]);
            $data = json_decode($response->getBody(), true);
            $this->accessToken = $data['data']['access_token'];
        } catch (RequestException $e) {
            throw new Exception('Failed to authenticate: ' . $e->getMessage());
        }
    }

    /**
     * Login to the OpenCart API
     * @return void
     */
    private function login() {
        try {
            $response = $this->client->post('login', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'json' => [
                    'username' => $_ENV['OPENCART_USERNAME'],
                    'password' => $_ENV['OPENCART_PASSWORD'],
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['success']) && !$data['success']) {
                throw new Exception('Failed to log in: ' . $data['error']);
            }
        } catch (RequestException $e) {
            throw new Exception('Failed to login: ' . $e->getMessage());
        }
    }

    /**
     * Get categories by level (default to level 2)
     * @param int $level
     * @return array
     */
    public function getCategoriesByLevel(int $level = 2) {
        try {
            $response = $this->client->get("categories/level/$level", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            // Decode the JSON response and return the `data` part
            $responseData = json_decode($response->getBody(), true);

            if (isset($responseData['data'])) {
                return $responseData['data']; // Return only the `data` part of the response
            } else {
                throw new Exception('Data key not found in response.');
            }
        } catch (RequestException $e) {
            throw new Exception('Failed to get categories: ' . $e->getMessage());
        }
    }

    /**
     * Fetch category name by ID using the API.
     *
     * @param int $categoryId
     * @return string
     * @throws Exception
     */
    public function fetchCategoryNameById(int $categoryId): string {
        try {
            $response = $this->client->get("categories/$categoryId", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            $responseData = json_decode($response->getBody(), true);

            if (isset($responseData['data'][0]['name'])) {
                return htmlspecialchars_decode($responseData['data'][0]['name'], ENT_QUOTES);
            } else {
                throw new Exception('Name key not found in response.');
            }
        } catch (RequestException $e) {
            throw new Exception('Failed to get category data: ' . $e->getMessage());
        }
    }


    /**
     * Get category ID by name
     * @param string $name
     * @return int|null
     */
    public function getCategoryIdByName(string $name) {
        try {
            $response = $this->client->get("categories/getcategoryidbyname/$name", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['success']) && $data['success'] && !empty($data['data'])) {
                return $data['data'][0]; // Assuming the first ID is the correct one
            }

            return null;
        } catch (RequestException $e) {
            throw new Exception('Failed to get category ID: ' . $e->getMessage());
        }
    }

    /**
     * Get category ID by name
     * @param string $name
     * @return int|null
     */
    public function getCategoryIdsByName(string $name) {
        try {
            $response = $this->client->get("categories/getcategoryidbyname/$name", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['success']) && $data['success'] && !empty($data['data'])) {
                return $data['data']; // Assuming the first ID is the correct one
            }

            return null;
        } catch (RequestException $e) {
            throw new Exception('Failed to get category ID: ' . $e->getMessage());
        }
    }

    /**
     * Add a category
     * @param array $categoryData
     * @return array
     */
    public function addCategory(array $categoryData) {
        try {
            $response = $this->client->post('categories', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'json' => $categoryData
            ]);

            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            throw new Exception('Failed to add category: ' . $e->getMessage());
        }
    }

    /**
     * Edit a category
     * @param int $categoryId
     * @param array $categoryData
     * @return array
     */
    public function editCategory(int $categoryId, array $categoryData) {
        try {
            $response = $this->client->put("categories/$categoryId", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'json' => $categoryData
            ]);

            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            throw new Exception('Failed to edit category: ' . $e->getMessage());
        }
    }

    /**
     * Delete a category by ID
     * @param int $categoryId
     * @return void
     */
    public function deleteCategory(int $categoryId) {
        try {
            $response = $this->client->delete("categories/$categoryId", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            if (isset($data['success']) && !$data['success']) {
                throw new Exception('Failed to delete category: ' . $data['error']);
            }
        } catch (RequestException $e) {
            throw new Exception('Failed to delete category: ' . $e->getMessage());
        }
    }

    /**
     * Check if an option value exists by name and return the option value ID if found
     * 
     * @param int $optionId The ID of the option (e.g., 13 for Printing Method)
     * @param string $optionValueName The name of the option value (e.g., "Screen Printed")
     * @param int $languageId The language ID (default is 1)
     * @return int|null The option value ID if the option value exists, null otherwise
     * @throws Exception If the request fails or any error occurs
     */
    public function isOptionValueNameExist(int $optionId, string $optionValueName, int $languageId = 1) {
        try {
            // Ensure option_value_name is URL-encoded to handle spaces and special characters
            $optionValueNameEncoded = urlencode($optionValueName);
    
            // Send GET request to the correct API endpoint for checking if the option value exists by name
            $response = $this->client->get("optionvalue", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'query' => [
                    'option_value_name' => $optionValueNameEncoded,
                    'language_id' => $languageId,
                    'id' => $optionId
                ]
            ]);
    
            // Decode the JSON response
            $data = json_decode($response->getBody(), true);
    
            // Check if the 'success' field is present and true, then return the option_value_id if available
            if (isset($data['success']) && $data['success'] && isset($data['data']['option_value_id'])) {
                return $data['data']['option_value_id'];
            }
    
            // Return null if option value does not exist
            return null;
            
        } catch (RequestException $e) {
            // Check if the response is 404, meaning option value doesn't exist
            if ($e->hasResponse()) {
                $statusCode = $e->getResponse()->getStatusCode();
                if ($statusCode == 404) {
                    // Option value doesn't exist, return null
                    return null;
                }
            }
    
            // For other exceptions, log the error and throw a new exception
            throw new Exception('Failed to check if option value exists: ' . $e->getMessage());
        }
    }    

    /**
     * Add multiple option values under specific option IDs asynchronously
     * @param array $optionValues An array of option values, where each entry includes:
     *                            - 'option_id': The ID of the option (e.g., 13 for Printing Method)
     *                            - 'name': The name of the option value (e.g., "Screen Printed")
     *                            - 'language_id': The language ID (default is 1)
     *                            - 'image': The image path for the option value (optional)
     *                            - 'sort_order': The sort order for the option value (optional)
     * @return array The response data from the API, including the IDs of added option values
     */
    public function addOptionValues(array $optionValues) {
        $results = [];
        $batches = array_chunk($optionValues, $_ENV['BATCH_SIZE']);

        foreach ($batches as $batch) {
            $promises = [];
            foreach ($batch as $optionValue) {
                // Set defaults if optional fields are not provided
                $optionId = $optionValue['option_id'];
                $optionValueName = $optionValue['name'];
                $languageId = $optionValue['language_id'] ?? 1;
                $image = $optionValue['image'] ?? '';
                $sortOrder = $optionValue['sort_order'] ?? 0;

                // Prepare the asynchronous request
                $promises[] = $this->client->postAsync("optionvalue/$optionId", [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $this->accessToken,
                    ],
                    'json' => [
                        'sort_order' => $sortOrder,
                        'image' => $image,
                        'option_value_description' => [
                            [
                                'name' => $optionValueName,
                                'language_id' => $languageId
                            ]
                        ]
                    ]
                ]);
            }

            // Wait for all requests to complete
            $responses = Promise\Utils::settle($promises)->wait();

            // Prepare the results array
            $result = [];
            foreach ($responses as $index => $response) {
                if ($response['state'] === 'fulfilled') {
                    $data = json_decode($response['value']->getBody(), true);
                    // Store the option_value_id if available, otherwise null
                    $result[] = [
                        'name' => $optionValues[$index]['name'],
                        'option_value_id' => $data['data']['option_value_id'] ?? null
                    ];
                } else {
                    // Log the error or throw an exception if needed
                    error_log("Failed to add option value for: {$optionValues[$index]['name']}. Reason: {$response['reason']}");
                    $result[] = [
                        'name' => $optionValues[$index]['name'],
                        'option_value_id' => null,
                        'error' => $response['reason']->getMessage()
                    ];
                }
            }
        }

        return $result;
    }

    /**
     * Retrieve a list of products from the OpenCart API.
     * This function fetches up to 10,000 products, returning essential details 
     * such as product ID, SKU, and model for each product.
     * 
     * @return array An array of product data containing 'product_id', 'sku', and 'model'.
     * @throws Exception If the API request fails or the data is not returned as expected.
     */
    public function getProductData() {
        try {
            $response = $this->client->get('products/limit/10000', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['success']) && $data['success'] && isset($data['data'])) {
                return $data['data'];
            } else {
                throw new Exception('Product data not found or request unsuccessful.');
            }
        } catch (RequestException $e) {
            throw new Exception('Failed to retrieve product data: ' . $e->getMessage());
        }
    }

    /**
     * Synchronously delete multiple products from the OpenCart API.
     * This function sends delete requests for each specified product ID sequentially.
     * 
     * @param array $productIds An array of product IDs to be deleted from OpenCart.
     * @return array An array of results for each deletion attempt, including status and any errors.
     */
    public function deleteProducts(array $productIds) {
        $results = [];
        $batches = array_chunk($productIds, $_ENV['BATCH_SIZE']);
    
        foreach ($batches as $batch) {
            $promises = [];
            foreach ($batch as $productId) {
                $promises[$productId] = $this->client->deleteAsync("products/$productId", [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $this->accessToken,
                    ]
                ]);
            }
    
            $responses = Promise\Utils::settle($promises)->wait();
            
            foreach ($responses as $productId => $response) {
                if ($response['state'] === 'fulfilled') {
                    $results[] = ['product_id' => $productId, 'status' => 'deleted'];
                } else {
                    $results[] = [
                        'product_id' => $productId,
                        'status' => 'failed',
                        'error' => $response['reason']->getMessage(),
                    ];
                }
            }
        }

        return $results;
    }



    /**
     * Get multiple product IDs by SKUs asynchronously
     * @param array $skus Array of SKUs to get product IDs for
     * @return array An array mapping SKU to product ID
     * @throws Exception
     */
    public function getProductIdsBySku(array $skus) {
        $skuToProductIdMap = [];
        $batches = array_chunk($skus, $_ENV['BATCH_SIZE']);
    
        foreach ($batches as $batch) {
            $promises = [];
            
            // Prepare asynchronous requests for all SKUs
            foreach ($batch as $sku) {
                $promises[$sku] = $this->client->getAsync("products/getproductidbysku/$sku", [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $this->accessToken,
                    ]
                ]);
            }

            // Wait for all requests to complete (fulfilled or rejected)
            $responses = Promise\Utils::settle($promises)->wait();

            // Prepare the results array
            foreach ($responses as $sku => $response) {
                if ($response['state'] === 'fulfilled') {
                    $data = json_decode($response['value']->getBody(), true);
                    // If the response is successful and contains data, store the product ID
                    if (isset($data['success']) && $data['success'] && !empty($data['data'])) {
                        $skuToProductIdMap[$sku] = $data['data']['id'];
                    } else {
                        // Product not found, set null
                        $skuToProductIdMap[$sku] = false;
                    }
                } elseif ($response['state'] === 'rejected') {
                    // Handle the error response
                    $reason = $response['reason'];
                    if ($reason instanceof RequestException && $reason->hasResponse() && $reason->getResponse()->getStatusCode() == 404) {
                        // If the product is not found (404), set null
                        $skuToProductIdMap[$sku] = false;
                    } else {
                        // Log the error for other types of failures
                        error_log("Failed to get product ID for SKU: {$sku}. Reason: {$reason->getMessage()}");
                        $skuToProductIdMap[$sku] = false;
                    }
                }
            }
        }

        return $skuToProductIdMap;
    }

    /**
     * Bulk add new products.
     * @param array $products Array of product data for bulk addition.
     * @return array API response.
     * @throws Exception
     */
    public function bulkAddProducts(array $products) {
        try {
            // Send POST request to bulk add products
            $response = $this->client->post('bulk_products', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'json' => $products // Send the products array as JSON
            ]);

            // Return the API response
            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            // Throw an exception if the API call fails
            throw new Exception('Failed to bulk add products: ' . $e->getMessage());
        }
    }

    /**
     * Bulk edit existing products.
     * @param array $products Array of product data for bulk editing (must include product_id).
     * @return array API response.
     * @throws Exception
     */
    public function bulkEditProducts(array $products) {
        try {
            // Send PUT request to bulk edit products
            $response = $this->client->put('bulk_products', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ],
                'json' => $products // Send the products array as JSON
            ]);

            // Return the API response
            return json_decode($response->getBody(), true);
        } catch (RequestException $e) {
            // Throw an exception if the API call fails
            throw new Exception('Failed to bulk edit products: ' . $e->getMessage());
        }
    }
}


